package org.msh.etbm.rest.authentication;

import org.jboss.resteasy.annotations.interception.ServerInterceptor;
import org.jboss.resteasy.core.Headers;
import org.jboss.resteasy.core.ResourceMethod;
import org.jboss.resteasy.core.ServerResponse;
import org.jboss.resteasy.spi.Failure;
import org.jboss.resteasy.spi.HttpRequest;
import org.jboss.resteasy.spi.interception.PostProcessInterceptor;
import org.jboss.resteasy.spi.interception.PreProcessInterceptor;
import org.jboss.seam.Component;
import org.jboss.seam.security.Identity;
import org.jboss.seam.web.Session;
import org.msh.etbm.services.auth.AuthenticationService;
import org.msh.tb.login.UserSession;

import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.Cookie;
import javax.ws.rs.ext.Provider;
import java.lang.reflect.Method;
import java.util.List;

/**
 * Authenticator interceptor, called when a rest call is made to check if user is authenticated
 * <p>
 * Created by ricardo on 03/12/14.
 */
@Provider
@ServerInterceptor
public class AuthenticatorInterceptor implements PreProcessInterceptor, PostProcessInterceptor {

    public static final String AUTH_TOKEN = "authToken";

    private static final ServerResponse UNAUTHORIZED =
            new ServerResponse("Access denied for this resource", 401, new Headers<Object>());
    ;

    private static final ServerResponse FORBIDDEN =
            new ServerResponse("Access denied for this resource", 403, new Headers<Object>());
    ;


    @Override
    public ServerResponse preProcess(HttpRequest req, ResourceMethod resourceMethod) throws Failure, WebApplicationException {
        Authenticated auth = getAuthenticationInfo(resourceMethod.getMethod());

        boolean authenticated = Identity.instance().isLoggedIn();
        UserSession.instance().setInvalidateSessionOnPostProcess(!authenticated);

        if (auth == null) {
            return null;
        }

        // if no session, try logging in with authToken
        if (!Identity.instance().isLoggedIn()) {
            ServerResponse res = loginWithAuthToken(req);
            if (res != null) {
                return res;
            }
        }

        // session already exists or if authentication is not required?
        if (!Identity.instance().isLoggedIn()) {
            // authenticate the user by the authentication token
            if (!authenticate(req)) {
                return UNAUTHORIZED;
            }
        }

        if (!checkPermission(auth.permissions())) {
            return FORBIDDEN;
        }

        return null;
    }

    /**
     * Check if user sent authentication token and if it is a valid one
     *
     * @param req
     * @return
     */
    private boolean authenticate(HttpRequest req) {
        // get auth token
        String authToken = null;

        Cookie cookie = req.getHttpHeaders().getCookies().get(AUTH_TOKEN);

        if (cookie != null) {
            authToken = cookie.getValue();
        } else {
            List<String> vals = req.getUri().getQueryParameters().get(AUTH_TOKEN);
            if ((vals != null) && (vals.size() > 0)) {
                authToken = vals.get(0);
            }
        }

        if (authToken == null) {
            return false;
        }

        AuthenticationService srv = (AuthenticationService) Component.getInstance("authenticationService");
        if (!srv.loginWithToken(authToken)) {
            return false;
        }

        return true;
    }

    @Override
    public void postProcess(ServerResponse serverResponse) {
        // check if session is closed by the end of the process
        if (UserSession.instance().isInvalidateSessionOnPostProcess()) {
            Session.instance().invalidate();
        }
    }

    private Authenticated getAuthenticationInfo(Method method) {
        Authenticated auth = method.getAnnotation(Authenticated.class);

        if (auth == null) {
            auth = method.getDeclaringClass().getAnnotation(Authenticated.class);
        }
        return auth;
    }

    /**
     * Check if user has permissions to go on with the request
     *
     * @param perms list of roles allowed for the request
     * @return true if user has the permissions necessary to continue
     */
    private boolean checkPermission(String[] perms) {
        if (perms == null || perms.length == 0) {
            return true;
        }

        for (String p : perms) {
            if (!Identity.instance().hasRole(p)) {
                return false;
            }
        }
        return true;
    }

    /**
     * Login and create session based on auth token
     * @param req
     * @return
     */
    private ServerResponse loginWithAuthToken(HttpRequest req) {
        // get auth token
        String authToken = null;

        Cookie cookie = req.getHttpHeaders().getCookies().get(AUTH_TOKEN);

        if (cookie != null) {
            authToken = cookie.getValue();
        } else {
            List<String> vals = req.getUri().getQueryParameters().get(AUTH_TOKEN);
            if ((vals != null) && (vals.size() > 0)) {
                authToken = vals.get(0);
            } else {
                List<String> lst = req.getHttpHeaders().getRequestHeader(AUTH_TOKEN);
                if (lst != null && lst.size() > 0) {
                    authToken = lst.get(0);
                }
            }
        }

        AuthenticationService srv = (AuthenticationService) Component.getInstance("authenticationService");
        if (!srv.loginWithToken(authToken)) {
            return UNAUTHORIZED;
        }

        return null;
    }
}
